﻿using log4net;
using Microsoft.VisualBasic.FileIO;
using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Net;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.TableService.Interface;
using VA.PPMS.IWS.TableService.Interface.Entities;
using VA.PPMS.IWS.ValidationDataService.Extensions;

namespace VA.PPMS.IWS.ValidationDataService.Data
{
    public class NppesData
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IPpmsContextHelper _ppmsContextHelper;
        private readonly IPpmsHelper _ppmsHelper;
        private readonly ITableService _tableService;

        private DateTime CurrentTargetDate { get; set; }
        
        public NppesData(ILog logger, IIwsConfiguration configuration, IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper, ITableService tableService)
        {
            _logger = logger;
            _configuration = configuration;
            _ppmsContextHelper = ppmsContextHelper;
            _ppmsHelper = ppmsHelper;
            _tableService = tableService;
            CurrentTargetDate = DateTime.Today;
        }

        public async Task ProcessCurrentData()
        {
            _logger.Info($"@@@@ INFO - Start Nppes Data at: {DateTime.Now} @@@@");

            var targetDate = await GetLastActivityDate();
            targetDate = targetDate.StartOfWeek(DayOfWeek.Monday);
            await StartRequest(targetDate);

            _logger.Info($"@@@@ INFO - End Nppes Data at: {DateTime.Now} @@@@");
        }

        private async Task StartRequest(DateTime targetDate)
        {
            CurrentTargetDate = targetDate;

            var uri = await GetCurrentUri(CurrentTargetDate);

            _logger.Info($"@@@@ Info - Requesting NPPES data @@@@");

            using (var client = new WebClient())
            {
                client.DownloadDataCompleted += Client_DownloadDataCompleted;
                client.DownloadDataAsync(uri);
            }
        }

        private async Task StartRequest()
        {
            var uri = await GetCurrentUri(CurrentTargetDate);

            _logger.Info($"@@@@ Info - Requesting NPPES data @@@@");

            using (var client = new WebClient())
            {
                client.DownloadDataCompleted += Client_DownloadDataCompleted;
                client.DownloadDataAsync(uri);
            }
        }

        private async Task StartNextRequest()
        {
            SetLastActivityDate().GetAwaiter().GetResult();

            var lastTargetDate = DateTime.Today.StartOfWeek(DayOfWeek.Monday);
            CurrentTargetDate = CurrentTargetDate.AddDays(7);

            if (CurrentTargetDate <= lastTargetDate)
            {
                await StartRequest();
            }
        }

        private void Client_DownloadDataCompleted(object sender, DownloadDataCompletedEventArgs e)
        {
            if (e.Error != null)
            {
                _logger.Info("---- Info: NPPES Data processing complete. ----");
            }
            else if (!e.Cancelled)
            {
                _logger.Info("@@@@ Info - Processing NPPES data @@@@");

                ProcessData(e.Result).GetAwaiter().GetResult();
                StartNextRequest().GetAwaiter().GetResult();
            }
            else
            {
                _logger.Info("@@@@ Info - No NPPES data to process yet @@@@");
            }
        }

        private async Task ProcessData(byte[] raw)
        {
            using (var compressedFileStream = new MemoryStream(raw))
            using (var zipArchive = new ZipArchive(compressedFileStream, ZipArchiveMode.Read, false))
            {
                foreach (var item in zipArchive.Entries)
                {
                    var fileName = item.Name;
                    // Skip all files that are not the NPI data file
                    if (Path.GetExtension(fileName) != ".csv" || !(fileName.Contains("npidata_") && !fileName.Contains("FileHeader"))) continue;

                    _logger.Info($"@@@@ Info - NPPES Data processing file: {fileName} @@@@");
                    using (var stream = item.Open())
                    {
                        await SaveFileAsync(stream);
                    }
                }
            }
        }

        private async Task<Uri> GetCurrentUri(DateTime targetDate)
        {
            const string datePattern = "MMddyy";

            var uri = await _configuration.GetNppesTargetUri();

            return new Uri(string.Format(uri, targetDate.ToString(datePattern), targetDate.AddDays(6).ToString(datePattern)));
        }

        private async Task SaveFileAsync(Stream inputStream)
        {
            const string delimit = ",";

            string npi;
            string npiType;
            string deactivationDate;
            string reactivationDate;
            string[] columns;
            string[] delimiter = new string[] { delimit };
            int i = 1;
            int actives = 0;
            string activeStatus = string.Empty;
            string key;

            var activeBatch = new List<NpiStatusEntity>();

            using (var parser = new TextFieldParser(inputStream))
            {
                parser.Delimiters = delimiter;
                while ((columns = parser.ReadFields()) != null)
                {
                    if (i > 1 && columns.Length > 40)
                    {
                        npi = CleanValue(columns[0]);
                        key = npi.Substring(0, 2);
                        npiType = columns[1];
                        deactivationDate = columns[39];
                        reactivationDate = columns[40];

                        if (string.IsNullOrEmpty(npiType)) npiType = "1";
                        activeStatus = CheckIsActive(deactivationDate, reactivationDate) ? _tableService.StatusActive : _tableService.StatusInactive;

                        activeBatch.Add(new NpiStatusEntity(npi, npiType, activeStatus));
                        actives++;

                        if (i % _tableService.SuperBatchSize == 0)
                        {
                            await _tableService.SaveNpiStatusBatchAsync(activeBatch);
                            activeBatch = new List<NpiStatusEntity>();

                            //if (i % (_tableService.SuperBatchSize * 2) == 0)
                            //{
                            //    // Save data for the current batch
                            //    _logger.Info($"---->> INFO - {DateTime.Now}> Saved batch [{i}] <<----");
                            //}
                        }
                    }

                    i++;
                    //if (i > 1001) break;
                }

                // Final update
                if (activeBatch != null && activeBatch.Count > 0)
                    await _tableService.SaveNpiStatusBatchAsync(activeBatch);
            }

            _logger.Info($"@@@@ Info - NPPES Data processing complete. @@@@");
        }

        public async Task QueryPpmsProviderIdentifersAsync(string npi, string npiType, string status)
        {
            using (var context = await _ppmsContextHelper.GetContextAsync())
            {
                var provIdentifier = context.ppms_provideridentifierSet.FirstOrDefault(pi => pi.ppms_ProviderIdentifier == npi && pi.ppms_Provider != null);
                if (provIdentifier != null)
                {
                    var provider = context.AccountSet.FirstOrDefault(p => p.Id == provIdentifier.ppms_Provider.Id);
                    
                    // Detach Provider from Context so we can make potential updates. 
                    context.Detach(provider);
                    
                    _logger.Info($"Provider Identifier in PPMS found for Provider: {provIdentifier.ppms_ProviderIdentifier}");

                    switch (npiType)
                    {
                        case "1":
                            // If the PPMS Provider Identifier NPI type does not match the Nppes Data, set PPMS Provider to Npi Check Failure. 
                            if (provIdentifier.ppms_IdentifierType.Value != (int)PpmsHelperEnums.ProviderNpiType.Individual)
                            {
                                // Deactivate Provider and set status to Npi Check Failure
                                await DeactivateRequest(context, provider);
                                break;
                            }
                            switch (status)
                            {
                                case "Active":
                                    // If the PPMS Provider Identifier Status is NOT Active and Nppes Data is Active, set the PPMS Provider to Active
                                    if (provIdentifier.StatusCode.Value != (int)PpmsHelperEnums.ProviderIdentifier_StatusCode.Active)
                                    {
                                        // Set Provider to Active if all other Validations true and set Nppes NPI field to true.
                                        await ActivateRequest(context, provider);
                                    }
                                    break;
                                case "Inactive":
                                    if (provIdentifier.StatusCode.Value != (int)PpmsHelperEnums.ProviderIdentifier_StatusCode.Inactive)
                                    {
                                        // Deactivate Provider and set status to Npi Check Failure
                                        await DeactivateRequest(context, provider);
                                    }
                                    break;
                            }
                            break;
                        case "2":
                            if (provIdentifier.ppms_IdentifierType.Value != (int)PpmsHelperEnums.ProviderNpiType.Organizational || provIdentifier.ppms_IdentifierType.Value != (int)PpmsHelperEnums.ProviderNpiType.GroupAgency)
                            {
                                // Deactivate Provider and set status to Npi Check Failure
                                await DeactivateRequest(context, provider);
                                break;
                            }
                            switch (status)
                            {
                                case "Active":
                                    if (provIdentifier.StatusCode.Value != (int)PpmsHelperEnums.ProviderIdentifier_StatusCode.Active)
                                    {
                                        // Set Provider to Active if all other Validations true and set Nppes NPI field to true.
                                        await ActivateRequest(context, provider);
                                    }
                                    break;

                                case "Inactive":
                                    if (provIdentifier.StatusCode.Value != (int)PpmsHelperEnums.ProviderIdentifier_StatusCode.Inactive)
                                    {
                                        // Deactivate Provider and set status to Npi Check Failure
                                        await DeactivateRequest(context, provider);
                                    }
                                    break;
                            }
                            break;
                    }
                }
            }
        }

        public async Task ActivateRequest(PpmsContext context, Account provider)
        {
            await Task.Run(() => {});

            // Check to see if all Validation fields on Provider are set to Active except for Nppes NPI.
            if (provider.ppms_addressvalidated == true && provider.ppms_providercontactvalidated == true && provider.ppms_geocoded == true && provider.ppms_leievalidated == true && provider.ppms_licensevalidated == true && provider.ppms_nppesnpivalidated != true)
            {
                context.Execute(_ppmsHelper.SetEntityStatus(provider, (int)PpmsHelperEnums.AccountState.Active, (int)PpmsHelperEnums.Account_StatusCode.Active));
            }
           
            // If there are other Validations not active, just set the NPPES field to true. 
            var updateProvider = new Account
            {
                Id = provider.Id,
                ppms_nppesnpivalidated = true
            };

            // Match sure Provider Update is attached to context. 
            if (!context.IsAttached(updateProvider)) context.Attach(updateProvider);
            context.UpdateObject(updateProvider);
            context.SaveChanges();
        }

        public async Task DeactivateRequest(PpmsContext context, Account provider)
        {
            await Task.Run(() => {});
            context.Execute(_ppmsHelper.SetEntityStatus(provider, (int)PpmsHelperEnums.AccountState.Inactive, (int)PpmsHelperEnums.Account_StatusCode.NpiCheckFailure));
            var updateProvider = new Account
            {
                Id = provider.Id,
                ppms_nppesnpivalidated = false
            };

            if (!context.IsAttached(updateProvider)) context.Attach(updateProvider);
            context.UpdateObject(updateProvider);
            context.SaveChanges();
        }

        private static bool CheckIsActive(string deactivate, string reactivate)
        {
            var d = ConvertToDate(deactivate);
            var r = ConvertToDate(reactivate);

            return (!d.HasValue || r.HasValue);
        }

        private static DateTime? ConvertToDate(string dateValue)
        {
            if (!string.IsNullOrEmpty(dateValue) && DateTime.TryParse(dateValue, out var theDate)) return theDate;
            return null;
        }

        private static string CleanValue(string value)
        {
            return value.Trim();
        }

        private async Task<DateTime> GetLastActivityDate()
        {
            var result = DateTime.Today;

            var activity = await _tableService.GetActivity(ActivityHistoryEntity.ActivityType.NPPES);
            if (activity != null && activity.ActivityDate.HasValue)
            {
                result = activity.ActivityDate.Value.AddDays(7);
            }

            return result;
        }

        private async Task SetLastActivityDate()
        {
            await _tableService.SetActivity(new ActivityHistoryEntity(ActivityHistoryEntity.ActivityType.NPPES, CurrentTargetDate));
        }
    }
}